/*
	File:		QuickDraw.h

	Contains:	C Interface to Quickdraw structures

	Copyright:	 1985-1996 by Apple Computer, Inc., all rights reserved.

	Derived from v37 internal.

*/


#ifndef __NEWTQD_H
#define __NEWTQD_H

#ifndef __CONFIGQD_H
#include "ConfigQD.h"
#endif

#ifndef __NEWTON_H
#include "Newton.h"
#endif



#ifndef __QUICKDRAW__
	enum
		{
	    /* Source transfer modes */
	    srcCopy = 0,
	    srcOr = 1,
	    srcXor = 2,
	    srcBic = 3,
	    notSrcCopy = 4,
	    notSrcOr = 5,
	    notSrcXor = 6,
	    notSrcBic = 7,
	    /* Pattern transfer modes */
	    patCopy = 8,
	    patOr = 9,
	    patXor = 10,
	    patBic = 11,
	    notPatCopy = 12,
	    notPatOr = 13,
	    notPatXor = 14,
	    notPatBic = 15
		};
#endif


	/*----- quickdraw structures -----*/

	typedef short StructSizeType;


	typedef FPoint point;
	typedef Fixed fixed;

	typedef struct curve
		{
		point 	first;
		point 	control;
		point 	last;
		} curve;
	typedef curve *curvePtr;

	typedef struct path
		{
		long	vectors;
		long	controlBits[1];
		point	vector[1];
		} path;

	typedef struct paths
		{
		long	contours;
		path	contour[1];
		} paths;
	typedef paths* pathsPtr;
	typedef pathsPtr* pathsHandle;

	typedef struct pathWalker
		{
		int isLine;
		curve c;
		/* private */
		long index;
		long ep;
		long* bits;
		point* p;
		} pathWalker;



	/*---  PixelMap ---*/

	#define		kPixMapStorage		0xC0000000	// to mask off the appropriate bits
	#define		kPixMapHandle		0x00000000	// baseAddr is a handle
	#define		kPixMapPtr			0x40000000	// baseAddr is a pointer
	#define		kPixMapOffset		0x80000000	// baseAddr is an offset from the PixelMap
	#define		kPixMapLittleEndian	0x20000000	// pixMap is little endian
	#define		kPixMapAllocated	0x10000000	// pixMap "owns" the bits memory
#ifdef QD_Gray
	#define		kPixMapGrayTable	0x08000000	// grayTable field exists
	#define		kPixMapNoPad		0x04000000	// direct pixel format, no pad byte
	#define		kPixMapByComponent	0x02000000	// direct pixel format, stored by component
	#define		kPixMapAntiAlias	0x01000000	// antialiasing ink text
#endif
	#define		kPixMapVersionMask	0x0000F000	// version of this struct
	#define		kPixMapDeviceType	0x00000F00	// bits 8..11 are device type code
	#define		kPixMapDevScreen	0x00000000	// 	screen or offscreen bitmap
	#define		kPixMapDevDotPrint	0x00000100	// 	dot matrix printer
	#define		kPixMapDevPSPrint	0x00000200	// 	postscript printer
	#define		kPixMapDepth		0x000000FF	// bits 0..7 are chunky pixel depth

	#define		kOneBitDepth		1
	#define 	kDefaultDPI			72			// default value for deviceRes fields
	#define 	kVersionShift		12

	#define		kPixMapVersion1		(0x0 << kVersionShift)
	#define		kPixMapVersion2		(0x1 << kVersionShift)

	typedef struct PixelMap
		{
		Ptr				baseAddr;
		short			rowBytes;		// pads to long
		Rect			bounds;
		ULong			pixMapFlags;
		Point			deviceRes;		// resolution of input device (0 indicates kDefaultDPI
#ifdef QD_Gray
		UChar*			grayTable;		// gray tone table
#endif
		} PixelMap;
	typedef PixelMap* PixelMapPtr;

#ifdef QD_CPlusPlus
#ifdef QD_Gray
		inline long GrayTableExists(const PixelMap* pixmap)	{ return pixmap->pixMapFlags & kPixMapGrayTable; }
		inline long NoPadByte(const PixelMap* pixmap)		{ return pixmap->pixMapFlags & kPixMapNoPad; }
		inline long ByComponent(const PixelMap* pixmap)		{ return pixmap->pixMapFlags & kPixMapByComponent; }
		inline long AntiAlias(const PixelMap* pixmap)		{ return pixmap->pixMapFlags & kPixMapAntiAlias; }
		inline long PixelDepth(const PixelMap* pixmap)		{ return pixmap->pixMapFlags & kPixMapDepth; }
#else
		inline long PixelDepth(PixelMap* /*pixmap*/)		{ return 1; }
#endif
		inline ULong IsPixMap(long rowBytes) 				{ return (rowBytes & 0x8000); }
		inline ULong GetPixMapVersion(const PixelMap* p)	{ return (p->pixMapFlags & kPixMapVersionMask); }
#endif	// QD_CPlusPlus


	/*---  Patterns ---*/

	typedef	PixelMap**	PatternHandle;						// patterns expressed as handle to PixelMap

	enum {whitePat, ltGrayPat, grayPat, dkGrayPat, blackPat};
	typedef unsigned char GetPatSelector;

	#define kPatternDataSize 		8						// number of pixels in square pattern definition
	#define kExpPatArraySize		16						// number of longs in expanded pattern array
	#define kExpPatArrayMask		kExpPatArraySize - 1



#ifndef __QUICKDRAW__
	typedef struct FontInfo
		{
		long	ascent;
		long	descent;
		long	widMax;
		long	leading;
		} FontInfo;
#endif




#ifndef __QUICKDRAW__
	typedef struct PenState
		{
		Point pnLoc;
		Point pnSize;
		short pnMode;
		PatternHandle fgPat;
		} PenState;

	typedef struct Region
		{
		StructSizeType	rgnSize;
		Rect	rgnBBox;
		} Region;
	typedef Region* RgnPtr;
	typedef RgnPtr* RgnHandle;

	typedef struct Picture
		{
		StructSizeType	picSize;
		Rect	picFrame;
		} Picture;
	typedef Picture* PicPtr;
	typedef PicPtr* PicHandle;

	typedef struct Polygon
		{
		StructSizeType	polySize;
		Rect	polyBBox;
		Point	polyPoints[1];
		} Polygon;
	typedef Polygon* PolyPtr;
	typedef PolyPtr* PolyHandle;
#endif		// __QUICKDRAW__




	/*--- Standard Procs ---*/

#ifndef __QUICKDRAW__
	enum {frame,paint,erase,invert,fill};
	typedef unsigned char GrafVerb;

	typedef void (*ArcProcPtr) (GrafVerb , Rect *, long , long );
	typedef void (*BitsProcPtr) (PixelMap *, Rect *, Rect *, long , RgnHandle );
	typedef void (*CurveProcPtr) (GrafVerb , curve * );
	typedef void (*GetPicDataProc) (Ptr , long );
	typedef void (*LineProcPtr) (Point );
	typedef void (*OvalProcPtr) (GrafVerb ,Rect *);
	typedef void (*PathsProcPtr) (GrafVerb , pathsHandle );
	typedef void (*PicCommentProc) (short ,short ,Handle );
	typedef void (*PolyProcPtr) (GrafVerb ,PolyHandle );
	typedef void (*PutPicDataProc) (Ptr , long );
	typedef void (*RectProcPtr) (GrafVerb ,Rect *);
	typedef void (*RgnProcPtr) (GrafVerb , RgnHandle );
	typedef void (*RRectProcPtr) (GrafVerb , Rect *, long , long );
	typedef void (*TextObjProc) (/*TextObjectRef*/ long , Fixed , Fixed );

	typedef struct QDProcs
		{
		ArcProcPtr			arcProc;
		BitsProcPtr			bitsProc;
		CurveProcPtr 		curveProc;
		GetPicDataProc		getPicProc;
		LineProcPtr			lineProc;
		OvalProcPtr			ovalProc;
		PathsProcPtr 		pathsProc;
		PicCommentProc		commentProc;
		PolyProcPtr			polyProc;
		PutPicDataProc		putPicProc;
		RectProcPtr			rectProc;
		RgnProcPtr			rgnProc;
		RRectProcPtr		rRectProc;
		TextObjProc			textProc;
		} QDProcs;
	typedef QDProcs* QDProcsPtr;


	typedef struct GrafPort
		{
		PixelMap		portBits;
		Rect			portRect;
		RgnHandle		visRgn;
		RgnHandle		clipRgn;
	    PatternHandle	fgPat;
	    PatternHandle	bgPat;
		Point			pnLoc;
		Point			pnSize;
		short			pnMode;
	    short			pnVis;
	    QDProcsPtr		grafProcs;

		Handle			picSave;		/* used for saving picts		*/
	    Handle			rgnSave;		/* used for saving rgns			*/
	    Handle			polySave;		/* used for saving poly's		*/
	    Point			patAlign;		/* used for picture playback	*/
		} GrafPort;
	typedef GrafPort* GrafPtr;

#ifdef QD_CPlusPlus
	inline long IsPrinterPort(const GrafPtr port)
		{ return port->portBits.pixMapFlags & (kPixMapDevDotPrint + kPixMapDevPSPrint); }
#endif

#endif		// __QUICKDRAW__


#endif
